<?php
namespace ShieldAIQ\Security;
if (!defined('ABSPATH')) { exit; }

final class StealthLogin {
    private static $license_cache = null;

    private static function gate_cookie_name(): string { return 'shieldaiq_gate'; }

    private static function issue_gate_cookie(): void {
        // Short-lived token: allows wp-login.php to run only for users who came via the secure URL.
        $token = wp_generate_password(24, false, false);
        set_transient('shieldaiq_gate_' . $token, time(), 5 * MINUTE_IN_SECONDS);

        // Cookie must be available for wp-login.php POST immediately.
        setcookie(self::gate_cookie_name(), $token, [
            'expires'  => time() + 5 * MINUTE_IN_SECONDS,
            'path'     => COOKIEPATH ? COOKIEPATH : '/',
            'domain'   => COOKIE_DOMAIN,
            'secure'   => is_ssl(),
            'httponly' => true,
            'samesite' => 'Lax',
        ]);
    }

    private static function has_valid_gate_cookie(): bool {
        $name = self::gate_cookie_name();
        $token = isset($_COOKIE[$name]) ? sanitize_text_field(wp_unslash($_COOKIE[$name])) : '';
        if ($token === '') return false;
        return get_transient('shieldaiq_gate_' . $token) !== false;
    }


    public static function init(): void {
        add_action('init',[__CLASS__,'add_rewrites'],5);
        add_action('template_redirect',[__CLASS__,'secure_login_entry'],0);

        add_action('init',[__CLASS__,'maybe_block_wp_login'],2);
        add_action('admin_init',[__CLASS__,'maybe_block_wp_admin_admin_init'],0);
    }

    public static function add_rewrites(): void {
        $s = Settings::get_settings();
        $slug = Utils::sanitize_slug($s['secure_login_slug'] ?? 'secure');
        add_rewrite_tag('%shieldaiq_secure_login%','1');
        add_rewrite_rule('^' . preg_quote($slug,'/') . '/?$', 'index.php?shieldaiq_secure_login=1', 'top');
    }

    public static function flush_rewrites(): void { self::add_rewrites(); flush_rewrite_rules(); }

    public static function secure_login_entry(): void {
        if (get_query_var('shieldaiq_secure_login') != 1) return;

        define('SHIELDAIQ_SECURE_LOGIN', true);

        // Allow wp-login.php only when coming from this secure URL.
        self::issue_gate_cookie();

        add_filter('login_headerurl', function(){ return 'https://shieldaiq.online'; });
        add_filter('login_headertext', function(){ return 'ShieldAIQ'; });
        add_filter('login_message', function($message){
            $s = Settings::get_settings();
            $logo = $s['logo_url'] ?? '';
            $html = '<div class="shieldaiq-login-top">';
            if ($logo) $html .= '<div class="shieldaiq-logo"><img src="'.esc_url($logo).'" alt="ShieldAIQ" /></div>';
            $html .= '<div class="shieldaiq-title">ShieldAIQ Secure Login</div>';
            $html .= '<div class="shieldaiq-sub">Use your private login URL only. Keep it secret.</div>';
            $html .= '</div>';
            return $html . $message;
        });

        add_action('login_enqueue_scripts', function(){
            echo '<style>
                body.login{background:#0b1220 !important}
                #login{width:380px}
                #loginform{border-radius:16px;background:#111a2e;border:1px solid rgba(255,255,255,.08);box-shadow:0 12px 34px rgba(0,0,0,.35)}
                .login label{color:#e6edf7}
                .login input[type=text],.login input[type=password]{border-radius:10px;border:1px solid rgba(255,255,255,.12);background:#0b1220;color:#e6edf7}
                .wp-core-ui .button-primary{background:#3b82f6;border-color:#3b82f6;box-shadow:none;text-shadow:none}
                .wp-core-ui .button-primary:hover{background:#2563eb;border-color:#2563eb}
                .login #nav a, .login #backtoblog a{color:#a9b8d3}
                .shieldaiq-login-top{text-align:center;margin:0 0 16px}
                .shieldaiq-login-top .shieldaiq-logo img{height:86px;width:auto;border-radius:16px}
                .shieldaiq-login-top .shieldaiq-title{color:#e6edf7;font-weight:900;font-size:18px;margin-top:10px}
                .shieldaiq-login-top .shieldaiq-sub{color:#a9b8d3;font-size:12px;margin-top:6px}
            </style>';
        });

        require_once ABSPATH . 'wp-login.php';
        exit;
    }

    public static function should_enforce_stealth(): bool {
        $s = Settings::get_settings();
        return !empty($s['stealth_enabled']) && self::is_license_active();
    }

    private static function is_recovery_request(): bool {
        $token = isset($_GET['shieldaiq_recover']) ? sanitize_text_field(wp_unslash($_GET['shieldaiq_recover'])) : '';
        if ($token==='') return false;
        $s = Settings::get_settings();
        return hash_equals((string)($s['recovery_token'] ?? ''), (string)$token);
    }

    public static function maybe_block_wp_login(): void {
        if (!self::should_enforce_stealth()) return;
        if (self::is_recovery_request()) return;

        $s = Settings::get_settings();
        if (empty($s['block_wp_login'])) return;

        $uri = $_SERVER['REQUEST_URI'] ?? '';
        if (is_string($uri) && strpos($uri,'wp-login.php') !== false) {

            // Allow wp-login ONLY if the request came through the secure login URL (gate cookie present).
            // This prevents brute-force bots from posting to wp-login.php directly.
            if (self::has_valid_gate_cookie()) {
                return;
            }

            Utils::log_event('blocked_wp_login','Blocked direct wp-login.php access (missing gate cookie).');
            status_header(404); nocache_headers(); exit;
        }
    }

    public static function maybe_block_wp_admin_admin_init(): void {
        if (!self::should_enforce_stealth()) return;
        if (is_user_logged_in()) return;
        if (self::is_recovery_request()) return;

        $s = Settings::get_settings();
        if (empty($s['block_wp_admin'])) return;

        if (defined('DOING_AJAX') && DOING_AJAX) return;

        Utils::log_event('blocked_wp_admin','Blocked wp-admin for logged-out user (stealth).');
        status_header(404); nocache_headers(); exit;
    }

    public static function is_license_active(): bool {
        if (self::$license_cache !== null) return (bool)self::$license_cache;
        $cached = get_transient('shieldaiq_license_active');
        if ($cached === '1') { self::$license_cache=true; return true; }
        if ($cached === '0') { self::$license_cache=false; return false; }
        self::$license_cache=false; return false;
    }

    public static function validate_best_effort(array $settings): void {
        $key = trim((string)($settings['license_key'] ?? ''));
        $server = trim((string)($settings['license_server'] ?? 'https://shieldaiq.online'));
        if ($key === '') { set_transient('shieldaiq_license_active','0',12*HOUR_IN_SECONDS); self::$license_cache=false; return; }

        $url = rtrim($server,'/') . '/wp-json/shieldaiq/v1/license/activate';
        $resp = wp_remote_post($url, [
            'timeout'=>10,
            'headers'=>['Content-Type'=>'application/json'],
            'body'=>wp_json_encode([
                'license_key'=>$key,
                'domain'=>Utils::site_host(),
                'product'=>'shieldaiq-security',
                'version'=>SHIELDAIQ_SEC_VERSION
            ]),
        ]);

        if (is_wp_error($resp)) {
            // Local test mode if server is unreachable.
            if (preg_match('/^SHIELDAIQ-[A-Z0-9]{4}(?:-[A-Z0-9]{4}){3,4}$/', strtoupper($key))) {
                set_transient('shieldaiq_license_active','1',6*HOUR_IN_SECONDS);
                self::$license_cache=true;
                Utils::log_event('license','License validated in local test mode (server unreachable).');
                return;
            }
            set_transient('shieldaiq_license_active','0',1*HOUR_IN_SECONDS);
            self::$license_cache=false;
            Utils::log_event('license','License activation failed (server unreachable).');
            return;
        }

        $code = wp_remote_retrieve_response_code($resp);
        $data = json_decode((string)wp_remote_retrieve_body($resp), true);
        if ($code === 200 && is_array($data) && !empty($data['valid'])) {
            set_transient('shieldaiq_license_active','1',12*HOUR_IN_SECONDS);
            self::$license_cache=true;
            Utils::log_event('license','License activated for domain.');
            return;
        }
        set_transient('shieldaiq_license_active','0',12*HOUR_IN_SECONDS);
        self::$license_cache=false;
        $reason = is_array($data) ? ($data['reason'] ?? 'invalid') : 'invalid';
        Utils::log_event('license','License invalid: '.$reason);
    }
}
