<?php
namespace ShieldAIQ\Security;
if (!defined('ABSPATH')) { exit; }

final class Settings {
    public static function init(): void {
        add_action('admin_menu',[__CLASS__,'menu']);
        add_action('admin_init',[__CLASS__,'register']);
        add_action('admin_enqueue_scripts',[__CLASS__,'enqueue']);
    }
    public static function menu(): void {
        add_options_page('ShieldAIQ Security','ShieldAIQ Security','manage_options','shieldaiq-security',[__CLASS__,'render']);
    }
    public static function enqueue($hook): void {
        if ($hook !== 'settings_page_shieldaiq-security') return;
        wp_enqueue_style('dashicons');
        wp_enqueue_style('shieldaiq-security-admin', SHIELDAIQ_SEC_URL.'assets/admin.css', [], SHIELDAIQ_SEC_VERSION);
    }
    public static function register(): void {
        register_setting('shieldaiq_security', Utils::opt_key('settings'), [
            'type'=>'array','sanitize_callback'=>[__CLASS__,'sanitize'],'default'=>[]
        ]);
    }
    public static function get_settings(): array {
        $s = get_option(Utils::opt_key('settings'), []);
        return is_array($s) ? $s : [];
    }
    public static function sanitize($input): array {
        $in = is_array($input) ? $input : [];
        $out = [];

        $out['hide_generator'] = !empty($in['hide_generator']) ? 1 : 0;
        $out['disable_xmlrpc'] = !empty($in['disable_xmlrpc']) ? 1 : 0;
        $out['remove_rsd'] = !empty($in['remove_rsd']) ? 1 : 0;
        $out['remove_wlw'] = !empty($in['remove_wlw']) ? 1 : 0;
        $out['remove_shortlink'] = !empty($in['remove_shortlink']) ? 1 : 0;

        $out['rate_limit_enabled'] = !empty($in['rate_limit_enabled']) ? 1 : 0;
        $out['rate_limit_max'] = isset($in['rate_limit_max']) ? max(1, min(30, intval($in['rate_limit_max']))) : 6;
        $out['rate_limit_window'] = isset($in['rate_limit_window']) ? max(1, min(240, intval($in['rate_limit_window']))) : 15;

        $out['protect_admin_logged_out'] = !empty($in['protect_admin_logged_out']) ? 1 : 0;
        $out['admin_block_mode'] = in_array(($in['admin_block_mode'] ?? 'login'), ['home','login','404'], true) ? $in['admin_block_mode'] : 'login';
        $out['license_key'] = isset($in['license_key']) ? sanitize_text_field($in['license_key']) : '';
        $out['license_server'] = isset($in['license_server']) ? esc_url_raw($in['license_server']) : 'https://shieldaiq.online';
        $out['log_enabled'] = !empty($in['log_enabled']) ? 1 : 0;
        $out['log_max'] = isset($in['log_max']) ? max(50, min(1000, intval($in['log_max']))) : 200;

        $out['weekly_email_enabled'] = !empty($in['weekly_email_enabled']) ? 1 : 0;
        $out['weekly_email_to'] = isset($in['weekly_email_to']) ? sanitize_email($in['weekly_email_to']) : '';
        $out['cache_page_enabled'] = !empty($in['cache_page_enabled']) ? 1 : 0;
        $out['cache_browser_headers'] = !empty($in['cache_browser_headers']) ? 1 : 0;
        $out['opt_minify_html'] = !empty($in['opt_minify_html']) ? 1 : 0;
        $out['opt_disable_emojis'] = !empty($in['opt_disable_emojis']) ? 1 : 0;
        $out['opt_defer_js'] = !empty($in['opt_defer_js']) ? 1 : 0;

        $out['recovery_token'] = isset($in['recovery_token']) ? sanitize_text_field($in['recovery_token']) : '';
        if ($out['recovery_token'] === '') $out['recovery_token'] = wp_generate_password(20, false, false);

        $old = self::get_settings();
        if (($old['secure_login_slug'] ?? '') !== $out['secure_login_slug']) {
            add_action('shutdown', function(){ \ShieldAIQ\Security\StealthLogin::flush_rewrites(); });
        }

        // Always validate after saving settings.
        add_action('shutdown', function() use ($out){ \ShieldAIQ\Security\StealthLogin::validate_best_effort($out); });

        return $out;
    }

    public static function render(): void {
        if (!current_user_can('manage_options')) return;
        $s = self::get_settings();

        // Clear logs
        if (isset($_POST['shieldaiq_clear_logs']) && check_admin_referer('shieldaiq_save_settings')) {
            update_option(Utils::opt_key('logs'), [], false);
            add_settings_error('shieldaiq_security', 'logs_cleared', 'Logs cleared.', 'updated');
            $s = self::get_settings();
        }

        // If key exists but cache is missing, validate now so the UI is accurate.
        if (!empty($s['license_key']) && get_transient('shieldaiq_license_active') === false) {
            \ShieldAIQ\Security\StealthLogin::validate_best_effort($s);
        }

        $license_active = \ShieldAIQ\Security\StealthLogin::is_license_active();
        $pro = $license_active;
        $name = Utils::opt_key('settings');
        $ai = get_transient('shieldaiq_ai_summary');
        ?>
        
        <div class="wrap shieldaiq-wrap">
            
        <style>
            .shieldaiq-topbar{display:flex;align-items:center;justify-content:space-between;gap:16px;margin:10px 0 18px 0;padding:12px 14px;background:#fff;border:1px solid #dcdcde;border-radius:12px;}
            .shieldaiq-brand{display:flex;align-items:center;gap:12px;min-width:240px;}
            .shieldaiq-brand img{height:36px;width:auto;display:block;}
            .shieldaiq-brand .shieldaiq-title2{font-size:20px;line-height:1.2;margin:0;}
            .shieldaiq-badges{display:flex;gap:8px;flex-wrap:wrap;margin-top:6px;}
            .shieldaiq-badge{display:inline-flex;align-items:center;gap:6px;padding:4px 10px;border-radius:999px;border:1px solid #dcdcde;background:#f6f7f7;font-weight:600;font-size:12px;}
            .shieldaiq-badge.pro{background:#eef6ff;border-color:#b6d6ff;}
            .shieldaiq-actions{display:flex;align-items:center;gap:10px;flex-wrap:wrap;}
            .shieldaiq-actions .button{margin:0;}
            .shieldaiq-features{display:grid;grid-template-columns:1fr 1fr;gap:14px;margin:10px 0 18px 0;}
            .shieldaiq-card{background:#fff;border:1px solid #dcdcde;border-radius:12px;padding:12px 14px;}
            .shieldaiq-card h3{margin:0 0 6px 0;}
            .shieldaiq-card ul{margin:0 0 0 18px;}
            @media (max-width: 900px){ .shieldaiq-features{grid-template-columns:1fr;} }
        </style>

            <div class="shieldaiq-topbar">
                <div class="shieldaiq-brand">
                    <img src="<?php echo esc_url('http://shieldaiq.online/wp-content/uploads/2025/12/shieldaiq.png'); ?>" alt="ShieldAIQ" />
                    <div>
                        <h1 class="shieldaiq-title2">ShieldAIQ Security</h1>
                        <div class="shieldaiq-badges">
                            <span class="shieldaiq-badge">Lite (Free)</span>
                            <?php if ($license_active) : ?>
                                <span class="shieldaiq-badge pro">Pro Active</span>
                            <?php else : ?>
                                <span class="shieldaiq-badge pro">Pro Locked</span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <div class="shieldaiq-actions">
                    <?php if (!$license_active) : ?>
                        <a class="button button-primary" href="<?php echo esc_url('https://shieldaiq.online/'); ?>" target="_blank" rel="noopener">Get Pro</a>
                    <?php else : ?>
                        <a class="button button-primary" href="<?php echo esc_url('https://shieldaiq.online/'); ?>" target="_blank" rel="noopener">shieldaiq.online</a>
                    <?php endif; ?>
                    <button type="submit" class="button button-primary">Save Changes</button>
                    <a class="button" href="<?php echo esc_url(add_query_arg('shieldaiq_purge', '1')); ?>">Purge cache / refresh rules</a>
                </div>
            </div>


            <p class="shieldaiq-sub">
                Free features run immediately. Premium AI reports unlock after license activation.
                <strong>Status:</strong> <?php echo $pro ? '<span class="ok">Active</span>' : '<span class="bad">Not active</span>'; ?>
            </p>
            <div class="shieldaiq-features">
                <div class="shieldaiq-card">
                    <h3>What you get in Lite (Free)</h3>
                    <ul>
                        <li>Hardening: hide generator tag, remove RSD/WLW/shortlink</li>
                        <li>Disable XML-RPC (optional)</li>
                        <li>Logs + security report</li>
                        <li>Performance &amp; Cache (safe): page cache (public), browser headers, HTML minify, emoji disable</li>
                    </ul>
                </div>
                <div class="shieldaiq-card">
                    <h3>Unlock with Pro license</h3>
                    <ul>
                        <li>AI weekly summary (server-powered) + recommendations</li>
                        <li>Pro-only performance: Defer JS toggle</li>
                        <li>More detailed “next steps” checklist</li>
                    </ul>
                </div>
            </div>


            <form method="post" action="options.php">
                <?php settings_fields('shieldaiq_security'); ?>

                <h2 class="shieldaiq-section"><?php echo Utils::dashicon('lock','c-purple'); ?> Hardening</h2>
                <table class="form-table" role="presentation">
                    <tr><th>Hide WordPress generator tag</th><td>
                        <label><input type="checkbox" name="<?php echo esc_attr($name); ?>[hide_generator]" value="1" <?php checked(!empty($s['hide_generator'])); ?> /> Enabled</label>
                        <p class="description">Removes the WordPress “generator” meta tag (very low risk).</p>
                    </td></tr>
                    <tr><th>Disable XML-RPC</th><td>
                        <label><input type="checkbox" name="<?php echo esc_attr($name); ?>[disable_xmlrpc]" value="1" <?php checked(!empty($s['disable_xmlrpc'])); ?> /> Enabled</label>
                        <p class="description">Blocks xmlrpc.php. Don’t enable if you use Jetpack or the WP mobile app.</p>
                    </td></tr>
                    <tr><th>Remove RSD link</th><td>
                        <label><input type="checkbox" name="<?php echo esc_attr($name); ?>[remove_rsd]" value="1" <?php checked(!empty($s['remove_rsd'])); ?> /> Enabled</label>
                        <p class="description">Removes the RSD discovery link (legacy). Minimal risk.</p>
                    </td></tr>
                    <tr><th>Remove WLW manifest</th><td>
                        <label><input type="checkbox" name="<?php echo esc_attr($name); ?>[remove_wlw]" value="1" <?php checked(!empty($s['remove_wlw'])); ?> /> Enabled</label>
                        <p class="description">Removes the Windows Live Writer manifest link (legacy). Minimal risk.</p>
                    </td></tr>
                    <tr><th>Remove shortlink output</th><td>
                        <label><input type="checkbox" name="<?php echo esc_attr($name); ?>[remove_shortlink]" value="1" <?php checked(!empty($s['remove_shortlink'])); ?> /> Enabled</label>
                        <p class="description">Removes WordPress shortlink tags/headers. Minimal risk.</p>
                    </td></tr>
                </table>

                <h2 class="shieldaiq-section"><?php echo Utils::dashicon('admin-network','c-teal'); ?> Admin protection</h2>
                <table class="form-table" role="presentation">
                    <tr><th>Protect wp-admin for logged-out users</th><td>
                        <label><input type="checkbox" name="<?php echo esc_attr($name); ?>[protect_admin_logged_out]" value="1" <?php checked(!empty($s['protect_admin_logged_out'])); ?> /> Enabled</label>
                        <p class="description">Controls what happens when logged-out visitors try /wp-admin/. (Stealth mode can force 404.)</p>
                    </td></tr>
                    <tr><th>When blocked, redirect to</th><td>
                        <select name="<?php echo esc_attr($name); ?>[admin_block_mode]">
                            <option value="login" <?php selected(($s['admin_block_mode'] ?? 'login'),'login'); ?>>Login page</option>
                            <option value="home" <?php selected(($s['admin_block_mode'] ?? 'login'),'home'); ?>>Homepage</option>
                            <option value="404" <?php selected(($s['admin_block_mode'] ?? 'login'),'404'); ?>>404 Not Found</option>
                        </select>
                        <p class="description">Recommended (non-stealth): Login page.</p>
                    </td></tr>
                    </table>

                <h2 class="shieldaiq-section"><?php echo Utils::dashicon('warning','c-red'); ?> Firewall-lite</h2>
                <table class="form-table" role="presentation">
                    <tr><th>Enable login rate limiting</th><td>
                        <label><input type="checkbox" name="<?php echo esc_attr($name); ?>[rate_limit_enabled]" value="1" <?php checked(!empty($s['rate_limit_enabled'])); ?> /> Enabled</label>
                        <p class="description">Limits failed login attempts by IP to slow down brute-force attacks.</p>
                    </td></tr>
                    <tr><th>Max login attempts</th><td>
                        <input type="number" min="1" max="30" name="<?php echo esc_attr($name); ?>[rate_limit_max]" value="<?php echo esc_attr($s['rate_limit_max'] ?? 6); ?>" />
                        <p class="description">Recommended 5–7.</p>
                    </td></tr>
                    <tr><th>Window (minutes)</th><td>
                        <input type="number" min="1" max="240" name="<?php echo esc_attr($name); ?>[rate_limit_window]" value="<?php echo esc_attr($s['rate_limit_window'] ?? 15); ?>" />
                        <p class="description">Counter reset window.</p>
                    </td></tr>
                </table>

                <h2 class="shieldaiq-section"><?php echo Utils::dashicon('admin-users','c-blue'); ?> License & Branding</h2>
                <table class="form-table" role="presentation">
                    <tr><th>License key</th><td>
                        <input type="text" class="regular-text" name="<?php echo esc_attr($name); ?>[license_key]" value="<?php echo esc_attr($s['license_key'] ?? ''); ?>" placeholder="SHIELDAIQ-XXXX-XXXX-XXXX-XXXX" />
                    </td></tr>
                    <tr><th>License server URL</th><td>
                        <input type="url" class="regular-text" name="<?php echo esc_attr($name); ?>[license_server]" value="<?php echo esc_attr($s['license_server'] ?? 'https://shieldaiq.online'); ?>" />
                    </td></tr>
                    </table>

                <h2 class="shieldaiq-section"><?php echo Utils::dashicon('clipboard','c-green'); ?> Logs</h2>
                <table class="form-table" role="presentation">
                    <tr><th>Enable logs</th><td>
                        <label><input type="checkbox" name="<?php echo esc_attr($name); ?>[log_enabled]" value="1" <?php checked(!empty($s['log_enabled'])); ?> /> Enabled</label>
                    </td></tr>
                    <tr><th>Max log entries</th><td>
                        <input type="number" min="50" max="1000" name="<?php echo esc_attr($name); ?>[log_max]" value="<?php echo esc_attr($s['log_max'] ?? 200); ?>" />
                    </td></tr>
                </table>

                <?php submit_button('Save Changes'); ?>

                <h2>Performance &amp; Cache (Security-Safe)</h2>
<p class="description">Optional, low-risk optimizations focused on security and stability. Page cache is for public pages only.</p>
<table class="form-table"><tbody>
<tr><th>Page cache (public pages)</th><td>
<label><input type="checkbox" name="<?php echo esc_attr(Utils::opt_key('settings')); ?>[cache_page_enabled]" value="1" <?php checked(!empty($s['cache_page_enabled'])); ?> /> Enabled</label>
<p class="description">Caches anonymous GET pages without query strings. Never caches logged-in/admin/REST/AJAX.</p>
</td></tr>
<tr><th>Browser cache headers</th><td>
<label><input type="checkbox" name="<?php echo esc_attr(Utils::opt_key('settings')); ?>[cache_browser_headers]" value="1" <?php checked(!empty($s['cache_browser_headers'])); ?> /> Enabled</label>
<p class="description">Adds lightweight Cache-Control headers for faster repeat visits and reduced bot load.</p>
</td></tr>
<tr><th>Minify HTML</th><td>
<label><input type="checkbox" name="<?php echo esc_attr(Utils::opt_key('settings')); ?>[opt_minify_html]" value="1" <?php checked(!empty($s['opt_minify_html'])); ?> /> Enabled</label>
<p class="description">Safe whitespace minification for frontend HTML (logged-out only).</p>
</td></tr>
<tr><th>Disable emojis</th><td>
<label><input type="checkbox" name="<?php echo esc_attr(Utils::opt_key('settings')); ?>[opt_disable_emojis]" value="1" <?php checked(!empty($s['opt_disable_emojis'])); ?> /> Enabled</label>
<p class="description">Removes emoji scripts/styles to reduce unnecessary JS requests.</p>
</td></tr>
<tr><th>Defer JavaScript (Pro)</th><td>
<label><input type="checkbox" name="<?php echo esc_attr(Utils::opt_key('settings')); ?>[opt_defer_js]" value="1" <?php checked(!empty($s['opt_defer_js'])); ?> <?php disabled(!\ShieldAIQ\Security\StealthLogin::is_license_active()); ?> /> Enabled</label>
<p class="description">Adds <code>defer</code> to non-critical scripts. Pro-only. (jQuery excluded)</p>
</td></tr>
</tbody></table>

<h3>ShieldAIQ Logs</h3>
<p><button type=\"submit\" name=\"shieldaiq_clear_logs\" class=\"button\">Clear logs</button></p>
<?php $logs = get_option(Utils::opt_key('logs'), []); if (!is_array($logs)) $logs = []; ?>
<?php if (empty($logs)) : ?><p><em>No logs yet.</em></p><?php else : ?>
<div style=\"max-height:360px; overflow:auto; border:1px solid #ccd0d4; background:#fff; padding:8px;\">
<table class=\"widefat striped\"><thead><tr><th style=\"width:160px;\">Time</th><th style=\"width:120px;\">Type</th><th>Message</th><th style=\"width:140px;\">IP</th><th>User-Agent</th></tr></thead><tbody>
<?php foreach ($logs as $row) { $ts=intval($row['ts']??0); $type=sanitize_text_field($row['type']??''); $msg=esc_html($row['msg']??''); $ip=esc_html($row['ip']??''); $ua=esc_html($row['ua']??''); echo '<tr><td>'.esc_html(date_i18n('Y-m-d H:i:s',$ts)).'</td><td><code>'.esc_html($type).'</code></td><td>'.$msg.'</td><td><code>'.$ip.'</code></td><td style=\"max-width:420px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap;\">'.$ua.'</td></tr>'; } ?>
</tbody></table></div><?php endif; ?>

<h2>Security Report</h2>
                <p class="description">Summary based on your recent logs (types, IPs, user agents) plus next-step recommendations.</p>
                <textarea class="large-text" rows="12" readonly><?php echo esc_textarea(self::build_security_report()); ?></textarea>

                <h3>Weekly email report (optional)</h3>
                <table class="form-table"><tbody>
                    <tr><th>Enable weekly email</th><td>
                        <label><input type="checkbox" name="<?php echo esc_attr(Utils::opt_key('settings')); ?>[weekly_email_enabled]" value="1" <?php checked(!empty($s['weekly_email_enabled'])); ?> /> Enabled</label>
                        <p class="description">Sends a weekly summary to the email below.</p>
                    </td></tr>
                    <tr><th>Send to</th><td>
                        <input type="email" class="regular-text" name="<?php echo esc_attr(Utils::opt_key('settings')); ?>[weekly_email_to]" value="<?php echo esc_attr($s['weekly_email_to'] ?? ''); ?>" placeholder="<?php echo esc_attr(get_option('admin_email')); ?>" />
                    </td></tr>
                    <tr><th>Test email</th><td>
                        <a class="button" href="<?php echo esc_url(wp_nonce_url(admin_url('admin-post.php?action=shieldaiq_ai_test_email'), 'shieldaiq_ai_test_email')); ?>">Send test email now</a>
                    </td></tr>
                </tbody></table>

            <?php submit_button('Save Changes'); ?>
</form>

            <hr />
            <h2 id="shieldaiq-ai"><?php echo Utils::dashicon('analytics','c-blue'); ?> AI Weekly Summary (server-powered)</h2>
            <?php if (!$pro): ?>
                <p><strong>AI summary requires an active license.</strong></p>
            <?php else: ?>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <input type="hidden" name="action" value="shieldaiq_ai_generate" />
                    <?php wp_nonce_field('shieldaiq_ai_generate'); ?>
                    <button class="button button-primary" type="submit">Generate / Refresh summary</button>
                <?php submit_button('Save Changes'); ?>
</form>

                <?php if (is_array($ai) && !empty($ai['summary'])): ?>
                    <div class="shieldaiq-ai-card">
                        <p><strong>Summary:</strong> <?php echo esc_html($ai['summary']); ?></p>
                        <?php if (!empty($ai['suspicious']) && is_array($ai['suspicious'])): ?>
                            <p><strong>Suspicious IP/User-Agent:</strong></p>
                            <ul>
                                <?php foreach (($ai['suspicious']['top_ips'] ?? []) as $x) echo '<li>IP: '.esc_html($x).'</li>'; ?>
                                <?php foreach (($ai['suspicious']['top_user_agents'] ?? []) as $x) echo '<li>UA: '.esc_html($x).'</li>'; ?>
                                <?php if (!empty($ai['suspicious']['notes'])) echo '<li>'.esc_html($ai['suspicious']['notes']).'</li>'; ?>
                            </ul>
                        <?php endif; ?>
                        <?php if (!empty($ai['suggested_rules']) && is_array($ai['suggested_rules'])): ?>
                            <p><strong>Suggested safe firewall rules:</strong></p><ul><?php foreach ($ai['suggested_rules'] as $x) echo '<li>'.esc_html($x).'</li>'; ?></ul>
                        <?php endif; ?>
                        <?php if (!empty($ai['fix_my_settings']) && is_array($ai['fix_my_settings'])): ?>
                            <p><strong>Fix my settings:</strong></p><ul><?php foreach ($ai['fix_my_settings'] as $x) echo '<li>'.esc_html($x).'</li>'; ?></ul>
                        <?php endif; ?>
                    </div>
                <?php elseif (is_array($ai) && !empty($ai['error'])): ?>
                    <p><strong>Error:</strong> <?php echo esc_html($ai['message'] ?? 'Unknown'); ?></p>
                <?php else: ?>
                    <p>No summary yet. Click “Generate / Refresh summary”.</p>
                <?php endif; ?>
            <?php endif; ?>

        </div>
        <?php
    }
    public static function build_security_report(): string {
        $s = self::get_settings();

        // Clear logs
        if (isset($_POST['shieldaiq_clear_logs']) && check_admin_referer('shieldaiq_save_settings')) {
            update_option(Utils::opt_key('logs'), [], false);
            add_settings_error('shieldaiq_security', 'logs_cleared', 'Logs cleared.', 'updated');
            $s = self::get_settings();
        }
        $logs = get_option(Utils::opt_key('logs'), []);
        if (!is_array($logs)) $logs = [];
        $logs = array_slice($logs, 0, 500);

        $counts = [];
        $ips = [];
        $uas = [];
        foreach ($logs as $e) {
            $t = (string)($e['type'] ?? 'unknown');
            $counts[$t] = ($counts[$t] ?? 0) + 1;
            $ip = (string)($e['ip'] ?? '');
            if ($ip) $ips[$ip] = ($ips[$ip] ?? 0) + 1;
            $ua = (string)($e['ua'] ?? '');
            if ($ua) $uas[$ua] = ($uas[$ua] ?? 0) + 1;
        }
        arsort($counts); arsort($ips); arsort($uas);

        $out = [];
        $out[] = 'ShieldAIQ Security Report';
        $out[] = 'Site: ' . Utils::site_host();
        $out[] = 'Generated: ' . date('Y-m-d H:i:s');
        $out[] = '';
        $out[] = 'Events summary by type:';
        if (!$counts) { $out[] = '- (no events logged yet)'; }
        foreach (array_slice($counts, 0, 12, true) as $k=>$v) {
            $out[] = "- {$k}: {$v}";
        }

        $out[] = '';
        $out[] = 'Top suspicious IPs:';
        if (!$ips) { $out[] = '- (none captured)'; }
        foreach (array_slice($ips, 0, 10, true) as $k=>$v) {
            $out[] = "- {$k} ({$v} events)";
        }

        $out[] = '';
        $out[] = 'Top User-Agents:';
        if (!$uas) { $out[] = '- (none captured)'; }
        foreach (array_slice($uas, 0, 6, true) as $k=>$v) {
            $short = substr($k, 0, 120);
            $out[] = "- {$short} ({$v})";
        }

        $out[] = '';
        $out[] = 'What to enable next:';
        if (empty($s['disable_xmlrpc'])) $out[] = "- Disable XML-RPC if you don't use Jetpack/mobile app.";
        if (empty($s['enable_rate_limiting'])) $out[] = "- Enable login rate limiting if you see brute-force attempts.";
        if (empty($s['log_enabled'])) $out[] = "- Enable logs to power reporting and AI summaries.";
        $out[] = "- Keep WordPress + plugins/themes updated, and use strong passwords + 2FA.";

        return implode("
", $out);
    }
}
