<?php
namespace ShieldAIQ\Security;
if (!defined('ABSPATH')) { exit; }

final class Cache {
    private const CACHE_DIR = 'shieldaiq/page';

    public static function init(): void {
        add_action('template_redirect', [__CLASS__, 'maybe_serve_page_cache'], 0);
        add_action('template_redirect', [__CLASS__, 'maybe_start_page_cache'], 1);
        add_action('send_headers', [__CLASS__, 'maybe_send_browser_cache_headers'], 0);

        add_action('init', [__CLASS__, 'maybe_disable_emojis'], 1);
        add_action('template_redirect', [__CLASS__, 'maybe_start_html_minify'], 2);

        add_filter('script_loader_tag', [__CLASS__, 'maybe_defer_js'], 10, 3);

        // Purge on content changes
        add_action('save_post', [__CLASS__, 'purge_all'], 10);
        add_action('deleted_post', [__CLASS__, 'purge_all'], 10);
        add_action('switch_theme', [__CLASS__, 'purge_all'], 10);
        add_action('upgrader_process_complete', [__CLASS__, 'purge_all'], 10, 0);
    }

    private static function s(): array { return Settings::get_settings(); }

    private static function should_optimize_frontend(): bool {
        if (is_admin()) return false;
        if (wp_doing_ajax()) return false;
        if (defined('REST_REQUEST') && REST_REQUEST) return false;
        return true;
    }

    private static function should_page_cache(): bool {
        $s = self::s();
        if (empty($s['cache_page_enabled'])) return false;

        if (!self::should_optimize_frontend()) return false;
        if (is_user_logged_in()) return false;

        // Only GET, no query string, no previews, no feeds
        $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        if (strtoupper($method) !== 'GET') return false;
        if (!empty($_GET)) return false;
        if (is_preview() || is_feed() || is_trackback()) return false;

        $uri = $_SERVER['REQUEST_URI'] ?? '';
        if (!is_string($uri)) return false;
        if (strpos($uri, '/wp-admin') !== false || strpos($uri, '/wp-login.php') !== false) return false;
        if (strpos($uri, 'wp-cron.php') !== false) return false;

        return true;
    }

    private static function cache_base_dir(): string {
        $base = trailingslashit(WP_CONTENT_DIR) . 'cache/';
        return $base . self::CACHE_DIR . '/';
    }

    private static function cache_key(): string {
        $scheme = is_ssl() ? 'https://' : 'http://';
        $host = $_SERVER['HTTP_HOST'] ?? parse_url(home_url(), PHP_URL_HOST);
        $uri  = $_SERVER['REQUEST_URI'] ?? '/';
        return hash('sha256', $scheme . $host . $uri);
    }

    private static function cache_file(): string {
        return self::cache_base_dir() . self::cache_key() . '.html';
    }

    private static function cache_meta_file(): string {
        return self::cache_base_dir() . self::cache_key() . '.json';
    }

    public static function maybe_serve_page_cache(): void {
        if (!self::should_page_cache()) return;

        $file = self::cache_file();
        if (!file_exists($file)) return;

        // Serve cached HTML
        nocache_headers(); // prevents WP from adding no-cache defaults in some cases
        header('X-ShieldAIQ-Cache: HIT');
        header('Content-Type: text/html; charset=' . get_bloginfo('charset'));
        readfile($file);
        exit;
    }

    public static function maybe_start_page_cache(): void {
        if (!self::should_page_cache()) return;

        // Start buffering to store HTML on shutdown
        ob_start(function($html){
            // Don't cache if empty or error-ish
            if (!is_string($html) || strlen($html) < 200) return $html;

            $dir = self::cache_base_dir();
            if (!is_dir($dir)) { wp_mkdir_p($dir); }

            $file = self::cache_file();
            $meta = self::cache_meta_file();
            @file_put_contents($file, $html);
            @file_put_contents($meta, wp_json_encode([
                'ts' => time(),
                'url' => (is_ssl() ? 'https://' : 'http://') . ($_SERVER['HTTP_HOST'] ?? '') . ($_SERVER['REQUEST_URI'] ?? '/'),
            ]));

            header('X-ShieldAIQ-Cache: MISS');
            return $html;
        });
    }

    public static function maybe_send_browser_cache_headers(): void {
        $s = self::s();
        if (empty($s['cache_browser_headers'])) return;
        if (!self::should_optimize_frontend()) return;

        // 7 days for static assets (best-effort; many are served by server anyway)
        // For HTML documents, keep it conservative.
        if (!headers_sent()) {
            header('Cache-Control: public, max-age=300'); // 5 minutes
        }
    }

    public static function maybe_disable_emojis(): void {
        $s = self::s();
        if (empty($s['opt_disable_emojis'])) return;

        remove_action('wp_head', 'print_emoji_detection_script', 7);
        remove_action('admin_print_scripts', 'print_emoji_detection_script');
        remove_action('wp_print_styles', 'print_emoji_styles');
        remove_action('admin_print_styles', 'print_emoji_styles');
        remove_filter('the_content_feed', 'wp_staticize_emoji');
        remove_filter('comment_text_rss', 'wp_staticize_emoji');
        remove_filter('wp_mail', 'wp_staticize_emoji_for_email');
    }

    public static function maybe_start_html_minify(): void {
        $s = self::s();
        if (empty($s['opt_minify_html'])) return;
        if (!self::should_optimize_frontend()) return;
        if (is_user_logged_in()) return;

        ob_start(function($html){
            if (!is_string($html)) return $html;
            // very safe minify: collapse whitespace between tags
            $html = preg_replace('/>\s+</', '><', $html);
            $html = preg_replace('/\s{2,}/', ' ', $html);
            return $html;
        });
    }

    public static function maybe_defer_js(string $tag, string $handle, string $src): string {
        $s = self::s();
        if (empty($s['opt_defer_js'])) return $tag;

        // Pro-only advanced toggle
        if (!StealthLogin::is_license_active()) return $tag;

        // Avoid deferring core critical libs
        $blocked = ['jquery', 'jquery-core', 'jquery-migrate'];
        if (in_array($handle, $blocked, true)) return $tag;

        if (strpos($tag, ' defer') === false) {
            $tag = str_replace('<script ', '<script defer ', $tag);
        }
        return $tag;
    }

    public static function purge_all(): void {
        $dir = self::cache_base_dir();
        if (!is_dir($dir)) return;
        foreach (glob($dir . '*') as $f) {
            @unlink($f);
        }
    }
}
