<?php
namespace ShieldAIQ\Security;
if (!defined('ABSPATH')) { exit; }

final class AI {
    public static function init(): void {
        add_action('admin_post_shieldaiq_ai_generate', [__CLASS__, 'generate']);
        add_action('admin_post_shieldaiq_purge_cache', [__CLASS__, 'purge_cache']);
        add_action('admin_post_shieldaiq_ai_test_email', [__CLASS__, 'test_email']);
        add_action('shieldaiq_weekly_email', [__CLASS__, 'send_weekly_email']);
        self::maybe_schedule_weekly();
    }

    public static function purge_cache(): void {
        if (!current_user_can('manage_options')) wp_die('Forbidden');
        check_admin_referer('shieldaiq_purge_cache');

        Utils::purge_caches();
        wp_safe_redirect(admin_url('options-general.php?page=shieldaiq-security&cache_purged=1'));
        exit;
    }


    private static function maybe_schedule_weekly(): void {
        $s = Settings::get_settings();
        $enabled = !empty($s['weekly_email_enabled']);
        $hook = 'shieldaiq_weekly_email';
        $next = wp_next_scheduled($hook);
        if ($enabled && !$next) {
            $ts = strtotime('next monday 09:00');
            if (!$ts) $ts = time() + WEEK_IN_SECONDS;
            wp_schedule_event($ts, 'weekly', $hook);
        } elseif (!$enabled && $next) {
            wp_clear_scheduled_hook($hook);
        }
    }

    public static function test_email(): void {
        if (!current_user_can('manage_options')) wp_die('Forbidden');
        check_admin_referer('shieldaiq_ai_test_email');
        self::send_weekly_email(true);
        wp_safe_redirect(admin_url('options-general.php?page=shieldaiq-security&email_sent=1'));
        exit;
    }

    public static function send_weekly_email(bool $is_test = false): void {
        $s = Settings::get_settings();
        if (!$is_test && empty($s['weekly_email_enabled'])) return;

        $to = sanitize_email($s['weekly_email_to'] ?? get_option('admin_email'));
        if (!$to) $to = get_option('admin_email');

        $report = Settings::build_security_report();
        $subject = $is_test ? 'ShieldAIQ Security Report (Test)' : 'ShieldAIQ Weekly Security Report';
        $headers = ['Content-Type: text/plain; charset=UTF-8'];
        wp_mail($to, $subject, $report, $headers);
    }

    public static function generate(): void {
        if (!current_user_can('manage_options')) wp_die('Forbidden');
        check_admin_referer('shieldaiq_ai_generate');

        $s = Settings::get_settings();
        $server = rtrim((string)($s['license_server'] ?? 'https://shieldaiq.online'), '/');
        $url = $server . '/wp-json/shieldaiq/v1/ai/weekly-summary';

        $logs = get_option(Utils::opt_key('logs'), []);
        if (!is_array($logs)) $logs = [];
        $logs = array_slice($logs, 0, 200);

        $payload = [
            'domain' => Utils::site_host(),
            'events' => array_map(function($e){
                return [
                    'ts' => intval($e['ts'] ?? 0),
                    'type' => sanitize_text_field($e['type'] ?? ''),
                    'ip' => sanitize_text_field($e['ip'] ?? ''),
                    'ua' => sanitize_text_field($e['ua'] ?? ''),
                    'msg' => sanitize_text_field($e['msg'] ?? ''),
                ];
            }, $logs),
        ];

        $resp = wp_remote_post($url, [
            'timeout' => 60,
            'redirection' => 2,
            'user-agent' => 'ShieldAIQ/' . SHIELDAIQ_SEC_VERSION,

            'timeout'=>20,
            'headers'=>['Content-Type'=>'application/json'],
            'body'=>wp_json_encode($payload),
        ]);

        if (is_wp_error($resp)) {
            set_transient('shieldaiq_ai_summary', ['error'=>true,'message'=>$resp->get_error_message() . ' (Tip: check outbound HTTPS from your server to ' . $server . ' and DNS/Firewall rules.)'], 15*MINUTE_IN_SECONDS);
        } else {
            $data = json_decode((string)wp_remote_retrieve_body($resp), true);
            if (!is_array($data)) $data = ['error'=>true,'message'=>'Invalid response'];
            set_transient('shieldaiq_ai_summary', $data, 12*HOUR_IN_SECONDS);
        }

        wp_safe_redirect(admin_url('options-general.php?page=shieldaiq-security#shieldaiq-ai'));
        exit;
    }
}
