<?php
/**
 * Plugin Name: Helpisto Chat Widget
 * Plugin URI: https://helpisto.com
 * Description: Add the Helpisto AI chat widget to your WordPress site in seconds.
 * Version: 1.0.0
 * Author: Helpisto
 * Author URI: https://helpisto.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: helpisto-chat-widget
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Helpisto_Chat_Widget {

    const OPTION_KEY = 'helpisto_chat_widget_settings';
    const SCRIPT_HANDLE = 'helpisto-chat-widget';

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'add_settings_page' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend' ] );
        add_filter( 'script_loader_tag', [ $this, 'add_data_attribute' ], 10, 3 );
        add_shortcode( 'helpisto_widget', [ $this, 'shortcode' ] );
    }

    public function add_settings_page() {
        add_options_page(
            'Helpisto Chat Widget',
            'Helpisto Chat Widget',
            'manage_options',
            'helpisto-chat-widget',
            [ $this, 'render_settings_page' ]
        );
    }

    public function register_settings() {
        register_setting(
            'helpisto_widget_group',
            self::OPTION_KEY,
            [
                'type' => 'array',
                'sanitize_callback' => [ $this, 'sanitize_settings' ],
                'default' => [
                    'enabled' => 1,
                    'helpisto_id' => '',
                    'load_everywhere' => 1,
                ],
            ]
        );
    }

    private function get_settings() {
        return wp_parse_args(
            get_option( self::OPTION_KEY, [] ),
            [
                'enabled' => 1,
                'helpisto_id' => '',
                'load_everywhere' => 1,
            ]
        );
    }

    private function is_uuid( $value ) {
        return (bool) preg_match(
            '/^[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{12}$/',
            $value
        );
    }

    public function sanitize_settings( $input ) {
        return [
            'enabled' => ! empty( $input['enabled'] ) ? 1 : 0,
            'load_everywhere' => ! empty( $input['load_everywhere'] ) ? 1 : 0,
            'helpisto_id' => isset( $input['helpisto_id'] ) && $this->is_uuid( trim( $input['helpisto_id'] ) )
                ? trim( $input['helpisto_id'] )
                : '',
        ];
    }

    public function render_settings_page() {
        $settings = $this->get_settings();
        ?>
        <div class="wrap">
            <h1>Helpisto Chat Widget</h1>
            <form method="post" action="options.php">
                <?php settings_fields( 'helpisto_widget_group' ); ?>
                <table class="form-table">
                    <tr>
                        <th>Enable widget</th>
                        <td><input type="checkbox" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[enabled]" value="1" <?php checked( $settings['enabled'], 1 ); ?>></td>
                    </tr>
                    <tr>
                        <th>Helpisto ID (UUID)</th>
                        <td>
                            <input type="text" class="regular-text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[helpisto_id]"
                                value="<?php echo esc_attr( $settings['helpisto_id'] ); ?>"
                                placeholder="ccf6ce0a-8211-4bcd-957b-574cdeeabddc">
                            <p class="description">
                                Sign up at <a href="https://helpisto.com" target="_blank" rel="noopener">helpisto.com</a>
                                and copy your Helpisto ID from your dashboard widget page.
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th>Load on every page</th>
                        <td><input type="checkbox" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[load_everywhere]" value="1" <?php checked( $settings['load_everywhere'], 1 ); ?>></td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
            <h2>Shortcode</h2>
            <code>[helpisto_widget]</code>
        </div>
        <?php
    }

    private function enqueue_script() {
        wp_register_script(
            self::SCRIPT_HANDLE,
            'https://helpisto.com/widget/helpisto-chat.js',
            [],
            null,
            true
        );
        wp_enqueue_script( self::SCRIPT_HANDLE );
    }

    public function enqueue_frontend() {
        if ( is_admin() ) return;
        $s = $this->get_settings();
        if ( empty( $s['enabled'] ) || empty( $s['helpisto_id'] ) ) return;
        if ( ! empty( $s['load_everywhere'] ) ) {
            $this->enqueue_script();
        }
    }

    public function add_data_attribute( $tag, $handle, $src ) {
        if ( self::SCRIPT_HANDLE !== $handle ) return $tag;
        $id = $this->get_settings()['helpisto_id'];
        return '<script src="' . esc_url( $src ) . '" data-helpisto-id="' . esc_attr( $id ) . '" async defer></script>';
    }

    public function shortcode() {
        $s = $this->get_settings();
        if ( empty( $s['enabled'] ) || empty( $s['helpisto_id'] ) ) return '';
        if ( empty( $s['load_everywhere'] ) ) {
            $this->enqueue_script();
        }
        return '';
    }
}

new Helpisto_Chat_Widget();
