<?php
if (!defined('ABSPATH')) exit;

require_once get_template_directory() . '/inc/section-engine.php';

function brandaiq_admin_assets($hook){
  if ($hook !== 'appearance_page_brandaiq-settings') return;
  wp_enqueue_style('brandaiq-admin', get_template_directory_uri().'/assets/admin/admin.css', [], '2.0.1');
  wp_enqueue_script('brandaiq-admin', get_template_directory_uri().'/assets/admin/admin.js', ['jquery'], '2.0.1', true);
  wp_enqueue_media();
}
add_action('admin_enqueue_scripts','brandaiq_admin_assets');

function brandaiq_register_settings(){
  register_setting('brandaiq_settings_group', 'brandaiq_company', 'brandaiq_sanitize_company');
}
add_action('admin_init','brandaiq_register_settings');

function brandaiq_sanitize_company($in){
  $def = brandaiq_defaults();
  $out = [];

  // simple text fields
  $text_keys = ['name','phone','email','address','hours','service_areas','footer_tagline','hero_title','hero_subtitle','hero_badge','hero_btn1_text','hero_btn1_url','hero_btn2_text','hero_btn2_url','call_banner_title','call_banner_text','call_banner_btn','helpisto_widget_id'];
  foreach($text_keys as $k){
    $v = $in[$k] ?? ($def[$k] ?? '');
    $out[$k] = sanitize_text_field($v);
  }
  // urls (relative allowed)
  $url_keys = ['hero_btn1_url','hero_btn2_url'];
  foreach($url_keys as $k){
    $v = $in[$k] ?? ($def[$k] ?? '');
    $out[$k] = sanitize_text_field($v);
  }

  // selects
  $mode = $in['color_mode'] ?? ($def['color_mode'] ?? 'yellow_black');
  $allowed_modes = ['yellow_black','navy_aqua','blue_orange','green_navy','red_black'];
  $out['color_mode'] = in_array($mode,$allowed_modes,true) ? $mode : 'yellow_black';

  // toggles
  $toggles = ['header_phone','floating_call','sec_hero','sec_services','sec_call_banner','sec_info','sec_why','sec_team','sec_testimonials','sec_map','sec_social','hero_use_slider','helpisto_enabled'];
  foreach($toggles as $k){
    $out[$k] = !empty($in[$k]) ? '1' : '0';
  }

  // hero slider autoplay
  $auto = absint($in['hero_slider_autoplay'] ?? ($def['hero_slider_autoplay'] ?? 5000));
  if ($auto < 2000) $auto = 2000;
  if ($auto > 20000) $auto = 20000;
  $out['hero_slider_autoplay'] = (string)$auto;

  // image ids
  $out['logo_id'] = (string) absint($in['logo_id'] ?? 0);
  $out['hero_bg_id'] = (string) absint($in['hero_bg_id'] ?? 0);
  $out['hero_plumber_id'] = (string) absint($in['hero_plumber_id'] ?? 0);

  // map embed (iframe only)
  $allowed = [
    'iframe' => [
      'src' => true,'width' => true,'height' => true,'style' => true,'allowfullscreen' => true,'loading' => true,'referrerpolicy' => true,
    ],
  ];
  $out['map_embed'] = wp_kses($in['map_embed'] ?? ($def['map_embed'] ?? ''), $allowed);

  // numeric
  $n = absint($in['testimonials_count'] ?? ($def['testimonials_count'] ?? 3));
  if (!in_array($n,[3,5], true)) $n = 3;
  $out['testimonials_count'] = (string)$n;

  // JSON lists: services/info/why/team/testimonials/social
  $out['services'] = brandaiq_clean_services_json($in['services'] ?? ($def['services'] ?? '[]'));
  $out['info_items'] = brandaiq_clean_title_text_json($in['info_items'] ?? ($def['info_items'] ?? '[]'));
  $out['why_items'] = brandaiq_clean_icon_title_text_json($in['why_items'] ?? ($def['why_items'] ?? '[]'));
  $out['team_items'] = brandaiq_clean_team_json($in['team_items'] ?? ($def['team_items'] ?? '[]'));
  $out['testimonials'] = brandaiq_clean_testimonials_json($in['testimonials'] ?? ($def['testimonials'] ?? '[]'));
  $out['social_items'] = brandaiq_clean_social_json($in['social_items'] ?? ($def['social_items'] ?? '[]'));

  // hero slides
  $out['hero_slides'] = brandaiq_clean_hero_slides_json($in['hero_slides'] ?? ($def['hero_slides'] ?? '[]'));

  return $out;
}

function brandaiq_clean_hero_slides_json($raw){
  $arr = json_decode(is_string($raw)?$raw:'[]', true);
  if (!is_array($arr)) $arr = [];
  $clean = [];
  foreach($arr as $it){
    if(!is_array($it)) continue;
    $clean[] = [
      'image_id' => (string) absint($it['image_id'] ?? 0),
      'title' => sanitize_text_field($it['title'] ?? ''),
      'subtitle' => sanitize_text_field($it['subtitle'] ?? ''),
      'badge' => sanitize_text_field($it['badge'] ?? ''),
      'btn1_text' => sanitize_text_field($it['btn1_text'] ?? ''),
      'btn1_url' => sanitize_text_field($it['btn1_url'] ?? ''),
      'btn2_text' => sanitize_text_field($it['btn2_text'] ?? ''),
      'btn2_url' => sanitize_text_field($it['btn2_url'] ?? ''),
      'text_color' => sanitize_hex_color($it['text_color'] ?? '') ?: '',
      'link_color' => sanitize_hex_color($it['link_color'] ?? '') ?: '',
      'button_bg' => sanitize_hex_color($it['button_bg'] ?? '') ?: '',
      'button_text' => sanitize_hex_color($it['button_text'] ?? '') ?: '',
      'button2_bg' => sanitize_hex_color($it['button2_bg'] ?? '') ?: '',
      'button2_text' => sanitize_hex_color($it['button2_text'] ?? '') ?: '',
      'badge_bg' => sanitize_hex_color($it['badge_bg'] ?? '') ?: '',
      'badge_text' => sanitize_hex_color($it['badge_text'] ?? '') ?: '',
    ];
  }
  // keep at least 1 slide so the UI doesn't break
  if (!$clean) $clean[] = ['image_id'=>'','title'=>'','subtitle'=>'','badge'=>'','btn1_text'=>'','btn1_url'=>'','btn2_text'=>'','btn2_url'=>'','text_color'=>'','link_color'=>'','button_bg'=>'','button_text'=>'','button2_bg'=>'','button2_text'=>'','badge_bg'=>'','badge_text'=>''];
  return wp_json_encode($clean);
}

function brandaiq_clean_services_json($raw){
  $arr = json_decode(is_string($raw)?$raw:'[]', true);
  if (!is_array($arr)) $arr = [];
  $clean = [];
  foreach($arr as $it){
    if(!is_array($it)) continue;
    $clean[] = [
      'icon' => sanitize_html_class($it['icon'] ?? 'dashicons-admin-tools'),
      'title' => sanitize_text_field($it['title'] ?? ''),
      'text' => sanitize_text_field($it['text'] ?? ''),
    ];
  }
  return wp_json_encode($clean);
}
function brandaiq_clean_title_text_json($raw){
  $arr = json_decode(is_string($raw)?$raw:'[]', true);
  if (!is_array($arr)) $arr = [];
  $clean = [];
  foreach($arr as $it){
    if(!is_array($it)) continue;
    $clean[] = [
      'title' => sanitize_text_field($it['title'] ?? ''),
      'text' => sanitize_text_field($it['text'] ?? ''),
    ];
  }
  return wp_json_encode($clean);
}
function brandaiq_clean_icon_title_text_json($raw){
  $arr = json_decode(is_string($raw)?$raw:'[]', true);
  if (!is_array($arr)) $arr = [];
  $clean = [];
  foreach($arr as $it){
    if(!is_array($it)) continue;
    $clean[] = [
      'icon' => sanitize_html_class($it['icon'] ?? 'dashicons-awards'),
      'title' => sanitize_text_field($it['title'] ?? ''),
      'text' => sanitize_text_field($it['text'] ?? ''),
    ];
  }
  return wp_json_encode($clean);
}
function brandaiq_clean_team_json($raw){
  $arr = json_decode(is_string($raw)?$raw:'[]', true);
  if (!is_array($arr)) $arr = [];
  $clean = [];
  foreach($arr as $it){
    if(!is_array($it)) continue;
    $clean[] = [
      'name' => sanitize_text_field($it['name'] ?? ''),
      'role' => sanitize_text_field($it['role'] ?? ''),
      'photo_id' => (string) absint($it['photo_id'] ?? 0),
    ];
  }
  return wp_json_encode($clean);
}
function brandaiq_clean_testimonials_json($raw){
  $arr = json_decode(is_string($raw)?$raw:'[]', true);
  if (!is_array($arr)) $arr = [];
  $clean = [];
  foreach($arr as $it){
    if(!is_array($it)) continue;
    $clean[] = [
      'quote' => sanitize_text_field($it['quote'] ?? ''),
      'name' => sanitize_text_field($it['name'] ?? ''),
    ];
  }
  return wp_json_encode($clean);
}
function brandaiq_clean_social_json($raw){
  $arr = json_decode(is_string($raw)?$raw:'[]', true);
  if (!is_array($arr)) $arr = [];
  $clean = [];
  foreach($arr as $it){
    if(!is_array($it)) continue;
    $clean[] = [
      'label' => sanitize_text_field($it['label'] ?? ''),
      'url' => esc_url_raw($it['url'] ?? ''),
    ];
  }
  return wp_json_encode($clean);
}

function brandaiq_add_menu(){
  add_theme_page('BrandAIQ Theme Settings', 'BrandAIQ Theme Settings', 'manage_options', 'brandaiq-settings', 'brandaiq_settings_page');
}
add_action('admin_menu','brandaiq_add_menu');

function brandaiq_settings_page(){
  if (!current_user_can('manage_options')) return;
  $opt = wp_parse_args((array)get_option('brandaiq_company', []), brandaiq_defaults());
  ?>
  <div class="wrap brandaiq-admin">
    <div class="brandaiq-admin-top">
      <div>
        <h1>BrandAIQ Theme Settings</h1>
        <div class="brandaiq-sub">Section engine • SEO • Mobile-first • Safe defaults</div>
      </div>
      <a class="button button-primary" href="https://helpisto.com/wordpress" target="_blank" rel="noopener">Download Helpisto Plugin</a>
    </div>

    <div class="brandaiq-tabs" role="tablist">
      <button class="brandaiq-tab is-active" data-tab="company">Company</button>
      <button class="brandaiq-tab" data-tab="design">Design</button>
      <button class="brandaiq-tab" data-tab="hero">Hero</button>
      <button class="brandaiq-tab" data-tab="services">Services</button>
      <button class="brandaiq-tab" data-tab="why">Why Choose Us</button>
      <button class="brandaiq-tab" data-tab="info">Info Highlights</button>
      <button class="brandaiq-tab" data-tab="team">Team</button>
      <button class="brandaiq-tab" data-tab="testimonials">Testimonials</button>
      <button class="brandaiq-tab" data-tab="map">Map</button>
      <button class="brandaiq-tab" data-tab="social">Social</button>
      <button class="brandaiq-tab" data-tab="helpisto">Helpisto</button>
    </div>

    <form method="post" action="options.php">
      <?php settings_fields('brandaiq_settings_group'); ?>

      <input type="hidden" id="brandaiq_services_json" name="brandaiq_company[services]" value="<?php echo esc_attr($opt['services']); ?>">
      <input type="hidden" id="brandaiq_why_json" name="brandaiq_company[why_items]" value="<?php echo esc_attr($opt['why_items']); ?>">
      <input type="hidden" id="brandaiq_info_json" name="brandaiq_company[info_items]" value="<?php echo esc_attr($opt['info_items']); ?>">
      <input type="hidden" id="brandaiq_team_json" name="brandaiq_company[team_items]" value="<?php echo esc_attr($opt['team_items']); ?>">
      <input type="hidden" id="brandaiq_testimonials_json" name="brandaiq_company[testimonials]" value="<?php echo esc_attr($opt['testimonials']); ?>">
      <input type="hidden" id="brandaiq_social_json" name="brandaiq_company[social_items]" value="<?php echo esc_attr($opt['social_items']); ?>">
      <input type="hidden" id="brandaiq_hero_slides_json" name="brandaiq_company[hero_slides]" value="<?php echo esc_attr($opt['hero_slides']); ?>">

      <div class="brandaiq-panels">

        <div class="brandaiq-panel is-active" data-panel="company">
          <div class="brandaiq-card">
            <h2>Company Info</h2>
            <div class="brandaiq-grid">
              <label>Company Name<input name="brandaiq_company[name]" value="<?php echo esc_attr($opt['name']); ?>"></label>

              <?php
                $logo_id = absint($opt['logo_id'] ?? 0);
                $logo_url = $logo_id ? wp_get_attachment_image_url($logo_id, 'medium') : '';
              ?>
              <div class="brandaiq-field brandaiq-span2">
                <div class="brandaiq-label">Logo</div>
                <div class="brandaiq-media">
                  <input type="hidden" id="brandaiq_logo_id" name="brandaiq_company[logo_id]" value="<?php echo esc_attr($logo_id); ?>">
                  <div class="brandaiq-media-preview" id="brandaiq_logo_preview">
                    <?php if($logo_url): ?>
                      <img src="<?php echo esc_url($logo_url); ?>" alt="Logo preview">
                    <?php else: ?>
                      <div class="brandaiq-media-empty">No logo selected</div>
                    <?php endif; ?>
                  </div>
                  <div class="brandaiq-media-actions">
                    <button type="button" class="button brandaiq-pick-logo">Choose Logo</button>
                    <button type="button" class="button brandaiq-remove-logo">Remove</button>
                    <div class="brandaiq-help">Recommended: transparent PNG/SVG, max 300×120px (uploads can be larger; we auto-fit).</div>
                  </div>
                </div>
              </div>

              <label>Phone<input name="brandaiq_company[phone]" value="<?php echo esc_attr($opt['phone']); ?>"></label>
              <label>Email<input name="brandaiq_company[email]" value="<?php echo esc_attr($opt['email']); ?>"></label>
              <label>Address<input name="brandaiq_company[address]" value="<?php echo esc_attr($opt['address']); ?>"></label>
              <label>Hours<input name="brandaiq_company[hours]" value="<?php echo esc_attr($opt['hours']); ?>"></label>
              <label>Service Areas<input name="brandaiq_company[service_areas]" value="<?php echo esc_attr($opt['service_areas']); ?>"></label>
              <label class="brandaiq-span2">Footer Tagline<input name="brandaiq_company[footer_tagline]" value="<?php echo esc_attr($opt['footer_tagline']); ?>"></label>
            </div>
          </div>

          <div class="brandaiq-card">
            <h2>Homepage Sections</h2>
            <div class="brandaiq-toggle-grid">
              <?php
              $secs = [
                'sec_hero' => 'Hero',
                'sec_services' => 'Services',
                'sec_call_banner' => 'Call Now Banner',
                'sec_info' => 'Info Highlights',
                'sec_why' => 'Why Choose Us',
                'sec_team' => 'Team',
                'sec_testimonials' => 'Testimonials',
                'sec_map' => 'Google Map',
                'sec_social' => 'Social Links',
              ];
              foreach($secs as $k=>$label): ?>
                <label class="brandaiq-toggle"><input type="checkbox" name="brandaiq_company[<?php echo esc_attr($k); ?>]" value="1" <?php checked($opt[$k],'1'); ?>><span><?php echo esc_html($label); ?></span></label>
              <?php endforeach; ?>
            </div>
          </div>
        </div>

        <div class="brandaiq-panel" data-panel="design">
          <div class="brandaiq-card">
            <h2>Design</h2>
            <div class="brandaiq-grid">
              <label>Color Mode
                <select name="brandaiq_company[color_mode]">
                  <option value="yellow_black" <?php selected($opt['color_mode'],'yellow_black'); ?>>Yellow / Black (Emergency)</option>
                  <option value="navy_aqua" <?php selected($opt['color_mode'],'navy_aqua'); ?>>Navy / Aqua</option>
                  <option value="blue_orange" <?php selected($opt['color_mode'],'blue_orange'); ?>>Blue / Orange</option>
                  <option value="green_navy" <?php selected($opt['color_mode'],'green_navy'); ?>>Green / Navy</option>
                  <option value="red_black" <?php selected($opt['color_mode'],'red_black'); ?>>Red / Black</option>
                </select>
              </label>
              <label class="brandaiq-toggle"><input type="checkbox" name="brandaiq_company[header_phone]" value="1" <?php checked($opt['header_phone'],'1'); ?>><span>Show phone button in header</span></label>
              <label class="brandaiq-toggle"><input type="checkbox" name="brandaiq_company[floating_call]" value="1" <?php checked($opt['floating_call'],'1'); ?>><span>Show floating Call Now button</span></label>
            </div>
          </div>
        </div>

        <div class="brandaiq-panel" data-panel="helpisto">
          <div class="brandaiq-card">
            <h2>Helpisto (Optional)</h2>
            <p>
              If you want AI chat on this website, install the Helpisto plugin and connect your Widget ID.
              This theme keeps Helpisto <strong>disabled by default</strong>.
            </p>
            <p>
              <a class="button button-primary" href="https://helpisto.com/wordpress" target="_blank" rel="noopener">Download Helpisto Plugin</a>
              <a class="button" href="https://helpisto.com" target="_blank" rel="noopener">Create Helpisto Account</a>
            </p>

            <div class="brandaiq-grid">
              <label class="brandaiq-toggle">
                <input type="checkbox" name="brandaiq_company[helpisto_enabled]" value="1" <?php checked($opt['helpisto_enabled'],'1'); ?>>
                <span>Enable Helpisto embed (adds script to footer)</span>
              </label>
              <label>Helpisto Widget ID
                <input name="brandaiq_company[helpisto_widget_id]" value="<?php echo esc_attr($opt['helpisto_widget_id']); ?>" placeholder="5774b002-1444-4ea3-b428-c1376de919f9">
                <small>Only the ID part (not the full &lt;script&gt; tag).</small>
              </label>
            </div>

            <div class="brandaiq-callout">
              <strong>Example:</strong>
              <code>&lt;script src="https://helpisto.com/widget/helpisto-chat.js" data-helpisto-id="YOUR-ID"&gt;&lt;/script&gt;</code>
            </div>
          </div>
        </div>

        <div class="brandaiq-panel" data-panel="hero">
          <div class="brandaiq-card">
            <h2>Hero</h2>
            <p class="brandaiq-muted">Tip: For best quality, upload wide images (recommended <strong>1600×700</strong> or larger). Use JPG/WEBP for photos.</p>
            <div class="brandaiq-grid">
              <label class="brandaiq-span2">Hero Title<input name="brandaiq_company[hero_title]" value="<?php echo esc_attr($opt['hero_title']); ?>"></label>
              <label class="brandaiq-span2">Hero Subtitle<textarea name="brandaiq_company[hero_subtitle]" rows="3"><?php echo esc_textarea($opt['hero_subtitle']); ?></textarea></label>
              <label class="brandaiq-span2">Top Badge Text<input name="brandaiq_company[hero_badge]" value="<?php echo esc_attr($opt['hero_badge']); ?>"></label>
              <label>Button 1 Text<input name="brandaiq_company[hero_btn1_text]" value="<?php echo esc_attr($opt['hero_btn1_text']); ?>"></label>
              <label>Button 1 URL (relative ok)<input name="brandaiq_company[hero_btn1_url]" value="<?php echo esc_attr($opt['hero_btn1_url']); ?>"></label>
              <label>Button 2 Text<input name="brandaiq_company[hero_btn2_text]" value="<?php echo esc_attr($opt['hero_btn2_text']); ?>"></label>
              <label>Button 2 URL (unused; Call uses phone)<input name="brandaiq_company[hero_btn2_url]" value="<?php echo esc_attr($opt['hero_btn2_url']); ?>"></label>
            </div>

            <div class="brandaiq-card brandaiq-subcard" style="margin-top:16px;">
              <h3>Hero Slider (optional)</h3>
              <div class="brandaiq-grid">
                <label class="brandaiq-toggle">
                  <input type="checkbox" name="brandaiq_company[hero_use_slider]" value="1" <?php checked($opt['hero_use_slider'],'1'); ?>>
                  <span>Enable slider on homepage hero</span>
                </label>
                <label>Autoplay (ms)
                  <input name="brandaiq_company[hero_slider_autoplay]" value="<?php echo esc_attr($opt['hero_slider_autoplay']); ?>" placeholder="5000">
                  <small>2000–20000 recommended.</small>
                </label>
              </div>

              <div id="brandaiq-hero-slides" class="brandaiq-repeater"></div>
              <p style="margin-top:10px;">
                <button type="button" class="button button-primary" id="brandaiq_add_slide">+ Add Slide</button>
                <button type="button" class="button" id="brandaiq_reset_slides">Reset to demo slides</button>
              </p>

              <div class="brandaiq-callout">
                <strong>How it works:</strong> Each slide has its own background image + text/buttons. The plumber cutout (right) stays the same.
              </div>

              <div class="brandaiq-hero-admin-preview">
                <strong>Quick Preview</strong>
                <div id="brandaiq_hero_preview_box" class="brandaiq-hero-preview-box">
                  <div class="brandaiq-hero-preview-inner">
                    <div class="brandaiq-hero-preview-card">
                      <div class="brandaiq-hero-preview-badge"></div>
                      <div class="brandaiq-hero-preview-title"></div>
                      <div class="brandaiq-hero-preview-sub"></div>
                      <div class="brandaiq-hero-preview-actions">
                        <span class="brandaiq-hero-preview-btn">Button 1</span>
                        <span class="brandaiq-hero-preview-btn dark">Call Now</span>
                      </div>
                    </div>
                    <div class="brandaiq-hero-preview-person" id="brandaiq_hero_preview_person"></div>
                  </div>
                </div>
                <small class="brandaiq-muted">For full preview, open your homepage in a new tab.</small>
              </div>
            </div>

            <div class="brandaiq-media-row">
              <div class="brandaiq-media">
                <h3>Hero Background</h3>
                <input type="hidden" id="brandaiq_hero_bg_id" name="brandaiq_company[hero_bg_id]" value="<?php echo esc_attr($opt['hero_bg_id']); ?>">
                <button type="button" class="button" id="brandaiq_pick_bg">Choose Image</button>
                <button type="button" class="button" id="brandaiq_clear_bg">Remove</button>
                <div class="brandaiq-preview" id="brandaiq_bg_preview">
                  <?php
                    $bg_id = absint($opt['hero_bg_id']);
                    if ($bg_id) {
                      $url = wp_get_attachment_image_url($bg_id, 'large');
                      if ($url) echo '<img src="'.esc_url($url).'" alt="" />';
                    }
                  ?>
                </div>
              </div>
              <div class="brandaiq-media">
                <h3>Plumber Cutout</h3>
                <input type="hidden" id="brandaiq_hero_plumber_id" name="brandaiq_company[hero_plumber_id]" value="<?php echo esc_attr($opt['hero_plumber_id']); ?>">
                <button type="button" class="button" id="brandaiq_pick_plumber">Choose Image</button>
                <button type="button" class="button" id="brandaiq_clear_plumber">Remove</button>
                <div class="brandaiq-preview" id="brandaiq_plumber_preview">
                  <?php
                    $pl_id = absint($opt['hero_plumber_id']);
                    if ($pl_id) {
                      $url = wp_get_attachment_image_url($pl_id, 'large');
                      if ($url) echo '<img src="'.esc_url($url).'" alt="" />';
                    }
                  ?>
                </div>
              </div>
            </div>
            <p class="brandaiq-help">Tip: Use a transparent PNG plumber cutout for best results.</p>


          </div>
        </div>

        <div class="brandaiq-panel" data-panel="services">
          <div class="brandaiq-card">
            <h2>Services Grid Editor</h2>
            <div id="brandaiq_services_builder"></div>
            <button type="button" class="button" id="brandaiq_add_service">+ Add Service</button>
            <p class="brandaiq-help">Icons use Dashicons class names (example: dashicons-admin-tools).</p>
          </div>
          <div class="brandaiq-card">
            <h2>Call Now Banner</h2>
            <div class="brandaiq-grid">
              <label class="brandaiq-span2">Title<input name="brandaiq_company[call_banner_title]" value="<?php echo esc_attr($opt['call_banner_title']); ?>"></label>
              <label class="brandaiq-span2">Text<input name="brandaiq_company[call_banner_text]" value="<?php echo esc_attr($opt['call_banner_text']); ?>"></label>
              <label>Button Label<input name="brandaiq_company[call_banner_btn]" value="<?php echo esc_attr($opt['call_banner_btn']); ?>"></label>
            </div>
          </div>
        </div>

        <div class="brandaiq-panel" data-panel="why">
          <div class="brandaiq-card">
            <h2>Why Choose Us (Editor)</h2>
            <div id="brandaiq_why_builder"></div>
            <button type="button" class="button" id="brandaiq_add_why">+ Add Item</button>
            <p class="brandaiq-help">Each item supports an icon (Dashicons), title, and text.</p>
          </div>
        </div>

        <div class="brandaiq-panel" data-panel="info">
          <div class="brandaiq-card">
            <h2>Info Highlights (Editor)</h2>
            <div id="brandaiq_info_builder"></div>
            <button type="button" class="button" id="brandaiq_add_info">+ Add Item</button>
          </div>
        </div>

        <div class="brandaiq-panel" data-panel="team">
          <div class="brandaiq-card">
            <h2>Team / Technicians</h2>
            <div id="brandaiq_team_builder"></div>
            <button type="button" class="button" id="brandaiq_add_team">+ Add Team Member</button>
            <p class="brandaiq-help">Photos use the WordPress Media Library.</p>
          </div>
        </div>

        <div class="brandaiq-panel" data-panel="testimonials">
          <div class="brandaiq-card">
            <h2>Testimonials</h2>
            <div class="brandaiq-grid">
              <label>How many to show
                <select name="brandaiq_company[testimonials_count]">
                  <option value="3" <?php selected($opt['testimonials_count'],'3'); ?>>3</option>
                  <option value="5" <?php selected($opt['testimonials_count'],'5'); ?>>5</option>
                </select>
              </label>
            </div>
            <div id="brandaiq_testimonials_builder"></div>
            <button type="button" class="button" id="brandaiq_add_testimonial">+ Add Testimonial</button>
          </div>
        </div>

        <div class="brandaiq-panel" data-panel="map">
          <div class="brandaiq-card">
            <h2>Google Map</h2>
            <label>Map embed code (iframe only)
              <textarea name="brandaiq_company[map_embed]" rows="5" style="width:100%;"><?php echo esc_textarea($opt['map_embed']); ?></textarea>
            </label>
          </div>
        </div>

        <div class="brandaiq-panel" data-panel="social">
          <div class="brandaiq-card">
            <h2>Social Links</h2>
            <div id="brandaiq_social_builder"></div>
            <button type="button" class="button" id="brandaiq_add_social">+ Add Social</button>
          </div>
        </div>

      </div>

      <?php submit_button('Save Changes'); ?>
      <!-- footer note intentionally hidden to keep UI clean -->
    </form>
  </div>
<?php } ?>
